/*
Half-Life MAP viewing utility.
Copyright (C) 2003  Ryan Samuel Gregg

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "stdafx.h"
#include "MainForm.h"

CMainForm::CMainForm(String *sStartupMAP, CConfig *Config)
{
	this->bCloseing = false;
	this->sStartupMAP = sStartupMAP;

	RecentFiles = new ArrayList();

	this->txtConsole = new ::CRichTextBox();
	
	this->Config = Config;
	this->Engine = new CEngine(Config, this->txtConsole);
	this->MAPLoader = new CMAPLoader(Config, this->txtConsole);
	this->PTSLoader = new CPTSLoader(Config, this->txtConsole);

	this->AboutForm = new CAboutForm("1.0.1", "Ryan Gregg", "02/09/2003", "C++", "5,668");

	this->mnuMainMenu = new ::MainMenu();

	this->cmdFile = new ::MenuItem("&File");
	this->cmdOpen = new ::MenuItem("&Open");
	this->cmdFileSep1 = new ::MenuItem("-");
	this->cmdRecentFiles = new ::MenuItem("&Recent Files");
	this->cmdFileSep2 = new ::MenuItem("-");
	this->cmdExit = new ::MenuItem("E&xit");

	this->cmdView = new ::MenuItem("&View");
	this->cmdTextured = new ::MenuItem("&Textured");
	this->cmdSolid = new ::MenuItem("&Solid");
	this->cmdWireFrame = new ::MenuItem("&Wireframe");
	this->cmdPoints = new ::MenuItem("&Points");
	this->cmdViewSep1 = new ::MenuItem("-");
	this->cmdDrawSpecialTextures = new ::MenuItem("Draw Special &Textures");
	this->cmdDrawSelection = new ::MenuItem("Draw &Selection");
	this->cmdDrawPointFile = new ::MenuItem("Draw &Point File");
	this->cmdViewSep2 = new ::MenuItem("-");
	this->cmdOutlineScene = new ::MenuItem("&Outline Scene");
	this->cmdLightScene = new ::MenuItem("&Light Scene");
	this->cmdViewSep3 = new ::MenuItem("-");
	this->cmdFog = new ::MenuItem("&Fog");
	this->cmdViewSep4 = new ::MenuItem("-");
	this->cmdTextureFilter = new ::MenuItem("Texture &Filter");
	this->cmdNonMipmapped = new ::MenuItem("&Non-Mipmapped");
	this->cmdMipmapped = new ::MenuItem("&Mipmapped");
	this->cmdNearest = new ::MenuItem("&Nearest");
	this->cmdLinear = new ::MenuItem("&Linear");
	this->cmdMipNearest = new ::MenuItem("&Nearest");
	this->cmdMipLinear = new ::MenuItem("&Linear");
	this->cmdMipBilinear = new ::MenuItem("&Bilinear");
	this->cmdMipTrilinear = new ::MenuItem("&Trilinear");
	this->cmdMipIsotropic = new ::MenuItem("&Isotropic");
	this->cmdMipAnisotropic = new ::MenuItem("&Anisotropic");

	this->cmdPointFile = new ::MenuItem("&Point File");
	this->cmdLoadPointFile = new ::MenuItem("&Load Point File");
	this->cmdUnloadPointFile = new ::MenuItem("&Unload Point File");

	this->cmdHelp = new ::MenuItem("&Help");
	this->cmdAbout = new ::MenuItem("&About");

	this->dlgOpenMAPFile = new ::OpenFileDialog();
	this->dlgOpenPTSFile = new ::OpenFileDialog();

	this->barStatusMain = new ::StatusBar();

	this->pnlSide = new ::Panel();
	this->pnlMain = new ::Panel();
	this->pnlBottom = new ::Panel();

	this->treWorld = new ::TreeView();
	this->splProperties = new ::Splitter();
	this->lstProperties = new ::ListView();

	this->splTree = new ::Splitter();

	this->pic3D = new ::PictureBox();

	this->splConsole = new ::Splitter();

	//this->txtConsole = new ::CRichTextBox();

	// Form Setup
	this->Text = S"MAP Viewer";
	this->AutoScaleBaseSize = ::Size(5, 13);
	this->ClientSize = ::Size(800, 600);
	this->StartPosition = ::FormStartPosition::CenterScreen;
	this->WindowState = ::FormWindowState::Normal;

	// Setup Menu
	this->mnuMainMenu->MenuItems->Add(this->cmdFile);
	this->mnuMainMenu->MenuItems->Add(this->cmdView);
	this->mnuMainMenu->MenuItems->Add(this->cmdPointFile);
	this->mnuMainMenu->MenuItems->Add(this->cmdHelp);

	this->cmdRecentFiles->Visible = false;
	this->cmdFileSep2->Visible = false;
	this->cmdFile->MenuItems->Add(this->cmdOpen);
	this->cmdFile->MenuItems->Add(this->cmdFileSep1);
	this->cmdFile->MenuItems->Add(this->cmdRecentFiles);
	this->cmdFile->MenuItems->Add(this->cmdFileSep2);
	this->cmdFile->MenuItems->Add(this->cmdExit);

	this->cmdTextured->RadioCheck = true;
	this->cmdSolid->RadioCheck = true;
	this->cmdWireFrame->RadioCheck = true;
	this->cmdPoints->RadioCheck = true;
	this->cmdView->MenuItems->Add(this->cmdTextured);
	this->cmdView->MenuItems->Add(this->cmdSolid);
	this->cmdView->MenuItems->Add(this->cmdWireFrame);
	this->cmdView->MenuItems->Add(this->cmdPoints);

	this->cmdLoadPointFile->Enabled = false;
	this->cmdUnloadPointFile->Enabled = false;
	this->cmdView->MenuItems->Add(this->cmdViewSep1);
	this->cmdView->MenuItems->Add(this->cmdDrawSpecialTextures);
	this->cmdView->MenuItems->Add(this->cmdDrawSelection);
	this->cmdView->MenuItems->Add(this->cmdDrawPointFile);
	this->cmdView->MenuItems->Add(this->cmdViewSep2);
	this->cmdView->MenuItems->Add(this->cmdOutlineScene);
	this->cmdView->MenuItems->Add(this->cmdLightScene);
	this->cmdView->MenuItems->Add(this->cmdViewSep3);
	this->cmdView->MenuItems->Add(this->cmdFog);
	this->cmdView->MenuItems->Add(this->cmdViewSep4);
	this->cmdView->MenuItems->Add(this->cmdTextureFilter);

	this->cmdPointFile->MenuItems->Add(this->cmdLoadPointFile);
	this->cmdPointFile->MenuItems->Add(this->cmdUnloadPointFile);

	this->cmdHelp->MenuItems->Add(this->cmdAbout);

	this->cmdTextureFilter->MenuItems->Add(this->cmdNonMipmapped);
	this->cmdTextureFilter->MenuItems->Add(this->cmdMipmapped);

	this->cmdNearest->RadioCheck = true;
	this->cmdLinear->RadioCheck = true;
	this->cmdNonMipmapped->MenuItems->Add(this->cmdNearest);
	this->cmdNonMipmapped->MenuItems->Add(this->cmdLinear);

	this->cmdMipNearest->RadioCheck = true;
	this->cmdMipLinear->RadioCheck = true;
	this->cmdMipBilinear->RadioCheck = true;
	this->cmdMipTrilinear->RadioCheck = true;
	this->cmdMipIsotropic->RadioCheck = true;
	this->cmdMipAnisotropic->RadioCheck = true;
	this->cmdMipmapped->MenuItems->Add(this->cmdMipNearest);
	this->cmdMipmapped->MenuItems->Add(this->cmdMipLinear);
	this->cmdMipmapped->MenuItems->Add(this->cmdMipBilinear);
	this->cmdMipmapped->MenuItems->Add(this->cmdMipTrilinear);
	this->cmdMipmapped->MenuItems->Add(this->cmdMipIsotropic);
	this->cmdMipmapped->MenuItems->Add(this->cmdMipAnisotropic);

	// Setup dialogs
	dlgOpenMAPFile->Title = "Open MAP File";
	dlgOpenMAPFile->Filter = "MAP Files (*.map)|*.map";

	dlgOpenPTSFile->Title = "Open PTS File";
	dlgOpenPTSFile->Filter = "PTS Files (*.pts)|*.pts";

	// Setup Status Bar
	this->barStatusMain->Dock = ::DockStyle::Bottom;

	// Setup Side Panel
	this->pnlSide->Dock = ::DockStyle::Left;
	this->pnlSide->Width = 160;

	// Setup Main Panel
	this->pnlMain->Dock = ::DockStyle::Fill;

	// Setup Bottom Panel
	this->pnlBottom->Dock = ::DockStyle::Bottom;
	this->pnlBottom->Height = 160;

	// Setup Tree
	this->treWorld->Dock = ::DockStyle::Top;
	this->treWorld->Height = 256;

	// Setup Property Splitter
	this->splProperties->Dock = ::DockStyle::Top;
	
	// Setup Properties
	this->lstProperties->Dock = ::DockStyle::Fill;
	this->lstProperties->FullRowSelect = true;
	this->lstProperties->GridLines = true;
	this->lstProperties->MultiSelect = false;
	this->lstProperties->View = System::Windows::Forms::View::Details;
	this->lstProperties->HeaderStyle = System::Windows::Forms::ColumnHeaderStyle::Nonclickable;

	ColumnHeader *Header;

	Header = new ColumnHeader();
	Header->Text = "Arg";
	this->lstProperties->Columns->Add(Header);

	Header = new ColumnHeader();
	Header->Text = "Val";
	this->lstProperties->Columns->Add(Header);

	// Setup Tree Splitter
	this->splTree->Dock = ::DockStyle::Left;

	// Setup Picture Box
	this->pic3D->Dock = ::DockStyle::Fill;
	this->pic3D->BackColor = ::SystemColors::ControlDark;

	// Setup Console Splitter
	this->splConsole->Dock = ::DockStyle::Bottom;
	
	// Setup Console
	this->txtConsole->Dock = ::DockStyle::Bottom;
	this->txtConsole->ReadOnly = true;
	
	// Add All Function Pointers
	this->add_Closing(new ::CancelEventHandler(this, &CMainForm::CMainForm_Closeing));
	this->add_Load(new ::EventHandler(this, &CMainForm::CMainForm_Load));

	this->pic3D->add_Resize(new ::EventHandler(this, &CMainForm::pic3D_Resize));
	this->pic3D->add_Paint(new ::PaintEventHandler(this, &CMainForm::pic3D_Paint));
	this->pic3D->add_MouseMove(new ::MouseEventHandler(this, &CMainForm::pic3D_MouseMove));
	this->pic3D->add_MouseDown(new ::MouseEventHandler(this, &CMainForm::pic3D_MouseDown));
	this->pic3D->add_MouseUp(new ::MouseEventHandler(this, &CMainForm::pic3D_MouseUp));
	this->pic3D->add_KeyDown(new ::KeyEventHandler(this, &CMainForm::pic3D_KeyDown));
	this->pic3D->add_KeyUp(new ::KeyEventHandler(this, &CMainForm::pic3D_KeyUp));

	this->treWorld->add_AfterSelect(new ::TreeViewEventHandler(this, &CMainForm::treWorld_AfterSelect));
	this->lstProperties->add_Resize(new ::EventHandler(this, &CMainForm::lstProperties_Resize));

	this->cmdOpen->add_Click(new ::EventHandler(this, &CMainForm::cmdOpen_Click));
	this->cmdExit->add_Click(new ::EventHandler(this, &CMainForm::cmdExit_Click));

	this->cmdTextured->add_Click(new ::EventHandler(this, &CMainForm::cmdTextured_Click));
	this->cmdSolid->add_Click(new ::EventHandler(this, &CMainForm::cmdSolid_Click));
	this->cmdWireFrame->add_Click(new ::EventHandler(this, &CMainForm::cmdWireFrame_Click));
	this->cmdPoints->add_Click(new ::EventHandler(this, &CMainForm::cmdPoints_Click));
	this->cmdDrawSpecialTextures->add_Click(new ::EventHandler(this, &CMainForm::cmdDrawSpecialTextures_Click));
	this->cmdDrawSelection->add_Click(new ::EventHandler(this, &CMainForm::cmdDrawSelection_Click));
	this->cmdDrawPointFile->add_Click(new ::EventHandler(this, &CMainForm::cmdDrawPointFile_Click));
	this->cmdOutlineScene->add_Click(new ::EventHandler(this, &CMainForm::cmdOutlineScene_Click));
	this->cmdLightScene->add_Click(new ::EventHandler(this, &CMainForm::cmdLightScene_Click));
	this->cmdFog->add_Click(new ::EventHandler(this, &CMainForm::cmdFog_Click));

	this->cmdNearest->add_Click(new ::EventHandler(this, &CMainForm::cmdNearest_Click));
	this->cmdLinear->add_Click(new ::EventHandler(this, &CMainForm::cmdLinear_Click));
	this->cmdMipNearest->add_Click(new ::EventHandler(this, &CMainForm::cmdMipNearest_Click));
	this->cmdMipLinear->add_Click(new ::EventHandler(this, &CMainForm::cmdMipLinear_Click));
	this->cmdMipBilinear->add_Click(new ::EventHandler(this, &CMainForm::cmdMipBilinear_Click));
	this->cmdMipTrilinear->add_Click(new ::EventHandler(this, &CMainForm::cmdMipTrilinear_Click));
	this->cmdMipIsotropic->add_Click(new ::EventHandler(this, &CMainForm::cmdMipIsotropic_Click));
	this->cmdMipAnisotropic->add_Click(new ::EventHandler(this, &CMainForm::cmdMipAnisotropic_Click));

	this->cmdLoadPointFile->add_Click(new ::EventHandler(this, &CMainForm::cmdLoadPointFile_Click));
	this->cmdUnloadPointFile->add_Click(new ::EventHandler(this, &CMainForm::cmdUnloadPointFile_Click));

	this->cmdAbout->add_Click(new ::EventHandler(this, &CMainForm::cmdAbout_Click));

	// Add All Controls
	this->pnlSide->Controls->Add(this->lstProperties);
	this->pnlSide->Controls->Add(this->splProperties);
	this->pnlSide->Controls->Add(this->treWorld);

	this->pnlMain->Controls->Add(this->pic3D);

	this->pnlBottom->Controls->Add(this->txtConsole);

	this->Menu = this->mnuMainMenu;
	this->Controls->Add(this->pnlMain);
	this->Controls->Add(this->splTree);
	this->Controls->Add(this->pnlSide);
	this->Controls->Add(this->splConsole);
	this->Controls->Add(this->txtConsole);
	this->Controls->Add(this->barStatusMain);
}

void CMainForm::LoadWorld(String *sFileName)
{
	if(!File::Exists(sFileName))
	{
		txtConsole->Print(String::Concat(sFileName, " not found.\n"), Color::Red);
		return;
	}

	CWorld *World = new CWorld(Config);
	CTextureManager *TextureManager = new CTextureManager(Config, this->txtConsole);

	if(this->MAPLoader->LoadMAPFile(sFileName, World, TextureManager))
	{
		this->Engine->SetWorld(World, TextureManager);

		txtConsole->Print("Building tree...", Color::Green);

		BuildTree(World);

		this->cmdLoadPointFile->Enabled = true;
		this->cmdUnloadPointFile->Enabled = false;

		AddRecentFile(sFileName);
		UpdateRecentFiles();

		this->barStatusMain->Text = String::Concat(sFileName, " loaded.");

		txtConsole->Print("done.\n", Color::Green);

		this->Engine->EnterRenderLoop();
	}

	GC::Collect();
}

void CMainForm::LoadPointFile(String *sFileName)
{
	CPointFile *PointFile;

	if(this->PTSLoader->LoadPTSFile(sFileName, &PointFile))
	{
		this->Engine->SetPointFile(PointFile);
		this->cmdUnloadPointFile->Enabled = true;
	}
}

void CMainForm::UnloadPointFile()
{
	this->Engine->SetPointFile(NULL);
	this->cmdUnloadPointFile->Enabled = false;
}

void CMainForm::BuildTree(CWorld *World)
{
	this->treWorld->BeginUpdate();
	this->treWorld->SelectedNode = NULL;
	this->treWorld->Nodes->Clear();
	this->lstProperties->Items->Clear();

	//Vertex3f Vertex;
	CFace *Face;
	CBrush *Brush;
	CEntity *Entity;
	CWorldObjectTreeNode *EntityNode, *BrushNode, *FaceNode;
	String *Name;
	for(int i = 0; i < World->GetEntities()->Count; i++)
	{
		Entity = static_cast<CEntity*>(World->GetEntities()->get_Item(i));
		Name = Entity->GetName();
		EntityNode = new CEntityTreeNode(Name, Entity);

		for(int j = 0; j < Entity->GetBrushes()->Count; j++)
		{
			Brush = static_cast<CBrush*>(Entity->GetBrushes()->get_Item(j));
			Name = String::Concat(S"brush ", j.ToString());
			BrushNode = new CBrushTreeNode(Name, Brush);

			for(int k = 0; k < Brush->GetFaces()->Count; k++)
			{
				Face = static_cast<CFace*>(Brush->GetFaces()->get_Item(k));
				Name = String::Concat(S"face ", k.ToString());
				FaceNode = new CFaceTreeNode(Name, Face);

				// Takes too much time...
				/*for(int l = 0; l < Face->GetVertexCount(); l++)
				{
					Vertex = Face->GetVertex(l);
					Name = String::Concat(Vertex.X.ToString(), S", ", Vertex.Y.ToString(), S", ", Vertex.Z.ToString());
					FaceNode->Nodes->Add(new TreeNode(Name));
				}*/

				BrushNode->Nodes->Add(FaceNode);
			}

			EntityNode->Nodes->Add(BrushNode);
		}

		this->treWorld->Nodes->Add(EntityNode);
	}

	this->treWorld->EndUpdate();
}

void CMainForm::AddRecentFile(String *sFile)
{
	for(int i = 0; i < RecentFiles->Count; i++)
	{
		if(static_cast<String*>(RecentFiles->get_Item(i))->ToLower()->Equals(sFile->ToLower()))
		{
			if(i == 0)
			{
				return;
			}
			else
			{
				RecentFiles->RemoveAt(i);
				RecentFiles->Insert(0, sFile);

				return;
			}
		}
	}

	RecentFiles->Insert(0, sFile);
}

void CMainForm::UpdateRecentFiles()
{
	::MenuItem *cmdRecentFile;
	cmdRecentFiles->MenuItems->Clear();

	while(RecentFiles->Count > Config->iRecentMAPCount)
	{
		RecentFiles->RemoveAt(RecentFiles->Count - 1);
	}

	for(int i = 0; i < RecentFiles->Count; i++)
	{
		cmdRecentFile = new ::MenuItem(static_cast<String*>(RecentFiles->get_Item(i)));
		cmdRecentFile->add_Click(new ::EventHandler(this, &CMainForm::cmdRecentFile_Click));
		this->cmdRecentFiles->MenuItems->Add(cmdRecentFile);
	}

	bool bDisplayRecentFiles = RecentFiles->Count != 0;
	this->cmdRecentFiles->Visible = bDisplayRecentFiles;
	this->cmdFileSep2->Visible = bDisplayRecentFiles;
}

void CMainForm::Exit()
{
	this->bCloseing = true;

	this->Engine->ExitRenderLoop();
	this->Engine->Destroy();

	txtConsole->Print("OpenGL destroyed.\n", Color::Green);

	WriteConfigFile(String::Concat(Application::StartupPath, "\\mv.ini"));

	this->txtConsole->SaveFile(String::Concat(Application::StartupPath, "\\log.txt"), RichTextBoxStreamType::PlainText);

	this->Close();
	Application::Exit();
}

void CMainForm::CMainForm_Load(::Object *sender, ::EventArgs *e)
{
	LoadConfigFile(String::Concat(Application::StartupPath, "\\mv.ini"));

	if(Config->sHalfLifePath->Length == 0)
	{
		GetHalfLifePath();
	}

	if(!this->Engine->Initialize((HWND)this->pic3D->Handle.ToInt32()))
	{
		txtConsole->Print("Error initializing OpenGL engine.\n", Color::Red);
	}
	else
	{
		txtConsole->Print("OpenGL initialized.\n", Color::Green);
	}

	this->Show();

	switch(Config->eRenderMode)
	{
		case RenderMode::Textured:
			cmdTextured->Checked = true;
			break;
		case RenderMode::Solid:
			cmdSolid->Checked = true;
			break;
		case RenderMode::WireFrame:
			cmdWireFrame->Checked = true;
			break;
		case RenderMode::Points:
			cmdPoints->Checked = true;
			break;
	}

	cmdDrawSpecialTextures->Checked = Config->bDrawSpecialTextures;

	cmdDrawSelection->Checked = Config->bDrawSelection;

	cmdDrawPointFile->Checked = Config->bDrawPointFile;

	cmdOutlineScene->Checked = Config->bOutlineScene;

	cmdLightScene->Checked = Config->bLightScene;

	cmdFog->Checked = Config->bFog;

	switch(Config->eTextureFilter)
	{
		case TextureFilter::Nearest:
			cmdNearest->Checked = true;
			break;
		case TextureFilter::Linear:
			cmdLinear->Checked = true;
			break;
		case TextureFilter::MipNearest:
			cmdMipNearest->Checked = true;
			break;
		case TextureFilter::MipLinear:
			cmdMipLinear->Checked = true;
			break;
		case TextureFilter::MipBilinear:
			cmdMipBilinear->Checked = true;
			break;
		case TextureFilter::MipTrilinear:
			cmdMipTrilinear->Checked = true;
			break;
		case TextureFilter::MipIsotropic:
			cmdMipIsotropic->Checked = true;
			break;
		case TextureFilter::MipAnisotropic:
			cmdMipAnisotropic->Checked = true;
			break;
	}

	this->Engine->Resize(this->pic3D->ClientSize.Width, this->pic3D->ClientSize.Height);

	if(sStartupMAP->Length != 0)
	{
		LoadWorld(sStartupMAP);
	}
}

void CMainForm::CMainForm_Closeing(::Object *sender, ::CancelEventArgs *e)
{
	if(!this->bCloseing)
	{
		e->Cancel = true;
		this->Exit();
	}
}

void CMainForm::pic3D_Resize(::Object *sender, ::EventArgs *e)
{
	this->Engine->Resize(this->pic3D->ClientSize.Width, this->pic3D->ClientSize.Height);
}

void CMainForm::pic3D_Paint(::Object *sender, ::PaintEventArgs *e)
{
	this->Engine->Paint();
}

void CMainForm::pic3D_MouseMove(::Object *sender, ::MouseEventArgs *e)
{
	this->Engine->SetMousePosition(e->get_X(), e->get_Y());
}

void CMainForm::pic3D_MouseDown(::Object *sender, ::MouseEventArgs *e)
{
	if(e->Button == MouseButtons::Left)
	{
		this->Engine->SetMousePressed(true);
	}

	this->pic3D->Focus();
}

void CMainForm::pic3D_MouseUp(::Object *sender, ::MouseEventArgs *e)
{
	if(e->Button == MouseButtons::Left)
	{
		this->Engine->SetMousePressed(false);
	}
}

void CMainForm::pic3D_KeyDown(::Object *sender, ::KeyEventArgs *e)
{
	Engine->SetKeyPressed(e->KeyCode, true);
}

void CMainForm::pic3D_KeyUp(::Object *sender, ::KeyEventArgs *e)
{
	Engine->SetKeyPressed(e->KeyCode, false);
}

void CMainForm::treWorld_AfterSelect(::Object *sender, ::TreeViewEventArgs *e)
{
	this->lstProperties->BeginUpdate();
	this->lstProperties->Items->Clear();

	if(this->treWorld->SelectedNode == NULL)
	{
		Engine->SetHighlightObject(NULL);
	}
	else
	{
		CWorldObjectTreeNode *WorldObjectTreeNode = static_cast<CWorldObjectTreeNode*>(this->treWorld->SelectedNode);
		CWorldObject *WorldObject = WorldObjectTreeNode->Selected();
		Engine->SetHighlightObject(WorldObject);

		if(WorldObjectTreeNode->GetType() == WorldObjectType::Entity)
		{
			CEntity *Entity = static_cast<CEntity*>(WorldObject);
			ArrayList *ArgVals;
			CArgVal *ArgVal;
			ListViewItem *Row;

			ArgVals = Entity->GetArgVals();

			for(int i = 0; i < ArgVals->Count; i++)
			{
				ArgVal = static_cast<CArgVal*>(ArgVals->get_Item(i));

				Row = new ListViewItem();
				Row->Text = ArgVal->GetArg();
				Row->SubItems->Add(ArgVal->GetVal());

				this->lstProperties->Items->Add(Row);
			}

			/*Row = new ListViewItem();
			Row->Text = "brushes";
			Row->SubItems->Add(Entity->GetBrushes()->Count.ToString());

			this->lstProperties->Items->Add(Row);*/
		}
		/*else if(WorldObjectTreeNode->GetType() == WorldObjectType::Brush)
		{
			ListViewItem *Row;
			CBrush *Brush = static_cast<CBrush*>(WorldObject);

			Row = new ListViewItem();
			Row->Text = "faces";
			Row->SubItems->Add(Brush->GetFaces()->Count.ToString());

			this->lstProperties->Items->Add(Row);
		}*/
		else if(WorldObjectTreeNode->GetType() == WorldObjectType::Face)
		{
			ListViewItem *Row;
			CFace *Face = static_cast<CFace*>(WorldObject);
			
			Row = new ListViewItem();
			Row->Text = "texture";
			Row->SubItems->Add(Face->GetTexture(Engine->GetTextureManager()));

			this->lstProperties->Items->Add(Row);

			Row = new ListViewItem();
			Row->Text = "shift";
			Row->SubItems->Add(String::Concat(Face->GetUShift().ToString("0.00"), ", ", Face->GetVShift().ToString("0.00")));

			this->lstProperties->Items->Add(Row);

			Row = new ListViewItem();
			Row->Text = "scale";
			Row->SubItems->Add(String::Concat(Face->GetUScale().ToString("0.00"), ", ", Face->GetVScale().ToString("0.00")));

			this->lstProperties->Items->Add(Row);

			Row = new ListViewItem();
			Row->Text = "rotation";
			Row->SubItems->Add(Face->GetRotation().ToString("0.00"));

			this->lstProperties->Items->Add(Row);

			/*Row = new ListViewItem();
			Row->Text = "vertices";
			Row->SubItems->Add(Face->GetVertexCount().ToString());

			this->lstProperties->Items->Add(Row);*/
		}
	}

	this->lstProperties->EndUpdate();
}

void CMainForm::lstProperties_Resize(::Object *sender, ::EventArgs *e)
{
	this->lstProperties->Columns->get_Item(0)->Width = this->lstProperties->ClientSize.Width / 2;
	this->lstProperties->Columns->get_Item(1)->Width = this->lstProperties->ClientSize.Width / 2;
}

void CMainForm::cmdOpen_Click(::Object *sender, ::EventArgs *e)
{
	if(dlgOpenMAPFile->ShowDialog() == DialogResult::OK)
	{
		LoadWorld(dlgOpenMAPFile->FileName);
	}
}

void CMainForm::cmdRecentFile_Click(::Object *sender, ::EventArgs *e)
{
	LoadWorld(static_cast<MenuItem*>(sender)->Text);
}

void CMainForm::cmdExit_Click(::Object *sender, ::EventArgs *e)
{
	this->Exit();
}

void CMainForm::cmdTextured_Click(::Object *sender, ::EventArgs *e)
{
	Config->eRenderMode = RenderMode::Textured;

	cmdTextured->Checked = true;
	cmdSolid->Checked = false;
	cmdWireFrame->Checked = false;
	cmdPoints->Checked = false;
}

void CMainForm::cmdSolid_Click(::Object *sender, ::EventArgs *e)
{
	Config->eRenderMode = RenderMode::Solid;

	cmdTextured->Checked = false;
	cmdSolid->Checked = true;
	cmdWireFrame->Checked = false;
	cmdPoints->Checked = false;
}

void CMainForm::cmdWireFrame_Click(::Object *sender, ::EventArgs *e)
{
	Config->eRenderMode = RenderMode::WireFrame;

	cmdTextured->Checked = false;
	cmdSolid->Checked = false;
	cmdWireFrame->Checked = true;
	cmdPoints->Checked = false;
}

void CMainForm::cmdPoints_Click(::Object *sender, ::EventArgs *e)
{
	Config->eRenderMode = RenderMode::Points;

	cmdTextured->Checked = false;
	cmdSolid->Checked = false;
	cmdWireFrame->Checked = false;
	cmdPoints->Checked = true;
}

void CMainForm::cmdDrawSpecialTextures_Click(::Object *sender, ::EventArgs *e)
{
	Config->bDrawSpecialTextures = ! Config->bDrawSpecialTextures;

	cmdDrawSpecialTextures->Checked = Config->bDrawSpecialTextures;
}

void CMainForm::cmdDrawSelection_Click(::Object *sender, ::EventArgs *e)
{
	Config->bDrawSelection = ! Config->bDrawSelection;

	cmdDrawSelection->Checked = Config->bDrawSelection;
}

void CMainForm::cmdDrawPointFile_Click(::Object *sender, ::EventArgs *e)
{
	Config->bDrawPointFile = ! Config->bDrawPointFile;

	cmdDrawPointFile->Checked = Config->bDrawPointFile;
}

void CMainForm::cmdOutlineScene_Click(::Object *sender, ::EventArgs *e)
{
	Config->bOutlineScene = ! Config->bOutlineScene;

	cmdOutlineScene->Checked = Config->bOutlineScene;
}

void CMainForm::cmdLightScene_Click(::Object *sender, ::EventArgs *e)
{
	Config->bLightScene = ! Config->bLightScene;

	cmdLightScene->Checked = Config->bLightScene;
}

void CMainForm::cmdFog_Click(::Object *sender, ::EventArgs *e)
{
	Config->bFog = ! Config->bFog;

	cmdFog->Checked = Config->bFog;
}

void CMainForm::cmdNearest_Click(::Object *sender, ::EventArgs *e)
{
	Config->eTextureFilter = TextureFilter::Nearest;

	this->cmdNearest->Checked = true;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdLinear_Click(::Object *sender, ::EventArgs *e)
{
	Config->eTextureFilter = TextureFilter::Linear;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = true;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdMipNearest_Click(::Object *sender, ::EventArgs *e)
{
	Config->eTextureFilter = TextureFilter::MipNearest;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = true;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdMipLinear_Click(::Object *sender, ::EventArgs *e)
{
	Config->eTextureFilter = TextureFilter::MipLinear;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = true;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdMipBilinear_Click(::Object *sender, ::EventArgs *e)
{
	Config->eTextureFilter = TextureFilter::MipBilinear;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = true;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdMipTrilinear_Click(::Object *sender, ::EventArgs *e)
{
	Config->eTextureFilter = TextureFilter::MipTrilinear;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = true;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdMipIsotropic_Click(::Object *sender, ::EventArgs *e)
{
	if(!Engine->TestExtension("GL_EXT_texture_filter_anisotropic"))
	{
		return;
	}

	Config->eTextureFilter = TextureFilter::MipIsotropic;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = true;
	this->cmdMipAnisotropic->Checked = false;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdMipAnisotropic_Click(::Object *sender, ::EventArgs *e)
{
	if(!Engine->TestExtension("GL_EXT_texture_filter_anisotropic"))
	{
		return;
	}

	Config->eTextureFilter = TextureFilter::MipAnisotropic;

	this->cmdNearest->Checked = false;
	this->cmdLinear->Checked = false;
	this->cmdMipNearest->Checked = false;
	this->cmdMipLinear->Checked = false;
	this->cmdMipBilinear->Checked = false;
	this->cmdMipTrilinear->Checked = false;
	this->cmdMipIsotropic->Checked = false;
	this->cmdMipAnisotropic->Checked = true;

	this->Engine->UpdateTextureFilter();
}

void CMainForm::cmdLoadPointFile_Click(::Object *sender, ::EventArgs *e)
{
	if(dlgOpenPTSFile->ShowDialog() == DialogResult::OK)
	{
		LoadPointFile(dlgOpenPTSFile->FileName);
	}
}

void CMainForm::cmdUnloadPointFile_Click(::Object *sender, ::EventArgs *e)
{
	UnloadPointFile();
}

void CMainForm::cmdAbout_Click(::Object *sender, ::EventArgs *e)
{
	AboutForm->ShowDialog();
}

void CMainForm::GetHalfLifePath()
{
	Object *HLPath = Config->ReadKey(Microsoft::Win32::Registry::LocalMachine, "SOFTWARE\\Valve\\Half-Life", "InstallPath");

	if(HLPath != NULL)
	{
		Config->sHalfLifePath = static_cast<String*>(HLPath);

		if(Config->sHalfLifePath->Length != 0 && !Config->sHalfLifePath->EndsWith("\\"))
		{
			Config->sHalfLifePath = String::Concat(Config->sHalfLifePath, "\\");
		}
	}
}

bool CMainForm::WriteConfigFile(String *sFile)
{
	txtConsole->Print(String::Concat(S"Writing ", sFile, S"..."), Color::Green);

	StreamWriter *ConfigFile;

	try
	{
		ConfigFile = new StreamWriter(sFile, false, System::Text::Encoding::ASCII);
	}
	catch(Exception *e)
	{
		txtConsole->Print(String::Concat(S"Error writing ", sFile, S": ", e->Message, S"\n"), Color::Red);
		return false;
	}

	try
	{
		ConfigFile->WriteLine("[General]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("General.RecentMAPCount = ", Config->iRecentMAPCount.ToString()));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Engine]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("Engine.ColorBits = ", Config->bColorBits.ToString()));
		ConfigFile->WriteLine(String::Concat("Engine.DepthBits = ", Config->bDepthBits.ToString()));

		ConfigFile->WriteLine(String::Concat("Engine.BackColor.R = ", Config->cBackColor.R.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.BackColor.G = ", Config->cBackColor.G.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.BackColor.B = ", Config->cBackColor.B.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.BackColor.A = ", Config->cBackColor.A.ToString("0.000")));

		ConfigFile->WriteLine(String::Concat("Engine.ForeColor.R = ", Config->cForeColor.R.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.ForeColor.G = ", Config->cForeColor.G.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.ForeColor.B = ", Config->cForeColor.B.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.ForeColor.A = ", Config->cForeColor.A.ToString("0.000")));

		ConfigFile->WriteLine(String::Concat("Engine.HighlightColor.R = ", Config->cHighlightColor.R.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.HighlightColor.G = ", Config->cHighlightColor.G.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.HighlightColor.B = ", Config->cHighlightColor.B.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.HighlightColor.A = ", Config->cHighlightColor.A.ToString("0.000")));

		ConfigFile->WriteLine(String::Concat("Engine.OutlineColor.R = ", Config->cOutlineColor.R.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.OutlineColor.G = ", Config->cOutlineColor.G.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.OutlineColor.B = ", Config->cOutlineColor.B.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Engine.OutlineColor.A = ", Config->cOutlineColor.A.ToString("0.000")));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Frustum]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("Frustum.FieldOfView = ", Config->fFrustumFieldOfView.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Frustum.ZNear = ", Config->fFrustumZNear.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Frustum.ZFar = ", Config->fFrustumZFar.ToString("0.000")));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Fog]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("Fog.Enabled = ", Config->bFog.ToString()));
		ConfigFile->WriteLine(String::Concat("Fog.Mode = ", __box(Config->eFogMode)->ToString()));
		ConfigFile->WriteLine(String::Concat("Fog.Density = ", Config->fFogDensity.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Fog.Start = ", Config->fFogStart.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Fog.End = ", Config->fFogEnd.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Fog.Color.R = ", Config->cFogColor.R.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Fog.Color.G = ", Config->cFogColor.G.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Fog.Color.B = ", Config->cFogColor.B.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Fog.Color.A = ", Config->cFogColor.A.ToString("0.000")));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Camera]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("Camera.Invert = ", Config->bInvertCamera.ToString()));
		ConfigFile->WriteLine(String::Concat("Camera.Speed = ", Config->fCameraSpeed.ToString("0.000")));
		ConfigFile->WriteLine(String::Concat("Camera.Boost = ", Config->fCameraBoost.ToString("0.000")));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Renderer]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("Renderer.DrawSpecialTextures = ", Config->bDrawSpecialTextures.ToString()));
		ConfigFile->WriteLine(String::Concat("Renderer.DrawSelection = ", Config->bDrawSelection.ToString()));
		ConfigFile->WriteLine(String::Concat("Renderer.DrawPointFile = ", Config->bDrawPointFile.ToString()));
		ConfigFile->WriteLine(String::Concat("Renderer.LightScene = ", Config->bLightScene.ToString()));
		ConfigFile->WriteLine(String::Concat("Renderer.OutlineScene = ", Config->bOutlineScene.ToString()));

		ConfigFile->WriteLine(String::Concat("Renderer.RenderMode = ", __box(Config->eRenderMode)->ToString()));
		ConfigFile->WriteLine(String::Concat("Renderer.TextureFilter = ", __box(Config->eTextureFilter)->ToString()));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Textures]");
		ConfigFile->WriteLine("");

		ConfigFile->WriteLine(String::Concat("Textures.HalfLifePath = ", Config->sHalfLifePath));

		ConfigFile->WriteLine("");
		ConfigFile->WriteLine("[Recent Files]");
		ConfigFile->WriteLine("");

		for(int i = RecentFiles->Count - 1; i >= 0; i--)
		{
			ConfigFile->WriteLine(String::Concat("RecentFiles.RecentFile = ", static_cast<String*>(RecentFiles->get_Item(i))));
		}
	}
	catch(Exception *e)
	{
		txtConsole->Print(String::Concat(S"Error writing ", sFile, S": ", e->Message, S"\n"), Color::Red);
		ConfigFile->Close();
		return false;
	}

	ConfigFile->Close();

	txtConsole->Print("done.\n", Color::Green);

	return true;
}

bool CMainForm::LoadConfigFile(String *sFile)
{
	if(!File::Exists(sFile))
		return false;

	txtConsole->Print(String::Concat(S"Loading ", sFile, S"..."), Color::Green);

	StreamReader *ConfigFile;

	try
	{
		ConfigFile = new StreamReader(sFile, System::Text::Encoding::ASCII);
	}
	catch(Exception *e)
	{
		txtConsole->Print(String::Concat(S"Error loading ", sFile, S": ", e->Message, S"\n"), Color::Red);
		return false;
	}

	try
	{
		String *sLine, *sArg, *sVal;

		while((sLine = ConfigFile->ReadLine()) != NULL)
		{
			if(!ProcessLine(sLine, &sArg, &sVal))
				continue;

			try
			{
				// General
				if(sArg->Equals("general.recentmapcount"))
				{
					Config->iRecentMAPCount = Convert::ToInt32(sVal);
				}

				// Engine
				else if(sArg->Equals("engine.colorbits"))
				{
					Config->bColorBits = Convert::ToByte(sVal);
				}
				else if(sArg->Equals("engine.depthbits"))
				{
					Config->bDepthBits = Convert::ToByte(sVal);
				}
				else if(sArg->Equals("engine.backcolor.r"))
				{
					Config->cBackColor.R = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.backcolor.g"))
				{
					Config->cBackColor.G = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.backcolor.b"))
				{
					Config->cBackColor.B = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.backcolor.a"))
				{
					Config->cBackColor.A = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.forecolor.r"))
				{
					Config->cForeColor.R = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.forecolor.g"))
				{
					Config->cForeColor.G = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.forecolor.b"))
				{
					Config->cForeColor.B = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.forecolor.a"))
				{
					Config->cForeColor.A = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.highlightcolor.r"))
				{
					Config->cHighlightColor.R = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.highlightcolor.g"))
				{
					Config->cHighlightColor.G = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.highlightcolor.b"))
				{
					Config->cHighlightColor.B = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.highlightcolor.a"))
				{
					Config->cHighlightColor.A = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.outlinecolor.r"))
				{
					Config->cOutlineColor.R = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.outlinecolor.g"))
				{
					Config->cOutlineColor.G = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.outlinecolor.b"))
				{
					Config->cOutlineColor.B = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("engine.outlinecolor.a"))
				{
					Config->cOutlineColor.A = Convert::ToSingle(sVal);
				}

				// Frustum
				else if(sArg->Equals("frustum.fieldofview"))
				{
					Config->fFrustumFieldOfView = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("frustum.znear"))
				{
					Config->fFrustumZNear = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("frustum.zfar"))
				{
					Config->fFrustumZFar = Convert::ToBoolean(sVal);
				}

				// Fog
				else if(sArg->Equals("fog.enabled"))
				{
					Config->bFog = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("fog.density"))
				{
					Config->fFogDensity = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.start"))
				{
					Config->fFogStart = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.end"))
				{
					Config->fFogEnd = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.color.r"))
				{
					Config->cFogColor.R = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.color.g"))
				{
					Config->cFogColor.G = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.color.b"))
				{
					Config->cFogColor.B = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.color.a"))
				{
					Config->cFogColor.A = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("fog.mode"))
				{
					Config->eFogMode = *static_cast<__box FogMode::FogMode*>(Enum::Parse(__typeof(FogMode::FogMode), sVal, false));
				}

				// Camera
				else if(sArg->Equals("camera.invert"))
				{
					Config->bInvertCamera = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("camera.speed"))
				{
					Config->fCameraSpeed = Convert::ToSingle(sVal);
				}
				else if(sArg->Equals("camera.boost"))
				{
					Config->fCameraBoost = Convert::ToSingle(sVal);
				}

				// Renderer
				else if(sArg->Equals("renderer.drawspecialtextures"))
				{
					Config->bDrawSpecialTextures = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("renderer.drawselection"))
				{
					Config->bDrawSelection = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("renderer.drawpointfile"))
				{
					Config->bDrawPointFile = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("renderer.lightscene"))
				{
					Config->bLightScene = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("renderer.outlinescene"))
				{
					Config->bOutlineScene = Convert::ToBoolean(sVal);
				}
				else if(sArg->Equals("renderer.rendermode"))
				{
					Config->eRenderMode = *static_cast<__box RenderMode::RenderMode*>(Enum::Parse(__typeof(RenderMode::RenderMode), sVal, false));
				}
				else if(sArg->Equals("renderer.texturefilter"))
				{
					Config->eTextureFilter = *static_cast<__box TextureFilter::TextureFilter*>(Enum::Parse(__typeof(TextureFilter::TextureFilter), sVal, false));
				}

				//Textures
				else if(sArg->Equals("textures.halflifepath"))
				{
					if(sVal->Length != 0 && !sVal->EndsWith("\\"))
					{
						sVal = String::Concat(sVal, "\\");
					}

					Config->sHalfLifePath = sVal;
				}

				// Recent Files
				else if(sArg->Equals("recentfiles.recentfile"))
				{
					AddRecentFile(sVal);
				}
			}
			catch(Exception *e)
			{
				continue;
			}
		}
	}
	catch(Exception *e)
	{
		txtConsole->Print(String::Concat(S"Error loading ", sFile, S": ", e->Message, S"\n"), Color::Red);
		ConfigFile->Close();
		return false;
	}

	ConfigFile->Close();

	UpdateRecentFiles();

	txtConsole->Print("done.\n", Color::Green);

	return true;
}

bool CMainForm::ProcessLine(String *sLine, String **sArg, String **sVal)
{
	*sArg = NULL;
	*sVal = NULL;

	sLine = sLine->Trim();

	if(sLine->IndexOf("//") != -1)
	{
		sLine = sLine->Remove(sLine->IndexOf("//"), sLine->Length)->Trim();

		if(sLine->Length == 0)
			return false;
	}

	if(sLine->IndexOf("=") == -1)
		return false;

	*sArg = sLine->Substring(0, sLine->IndexOf("="))->Trim()->ToLower();
	*sVal = sLine->Substring(sLine->IndexOf("=") + 1)->Trim();

	return true;
}